classdef TriangleCovTree <  handle
    % TriangleCovTree - used to speed up search for closest point
    % 
    % Tree = TriangleCovTree(Mesh)
    %
    % [cp,ct,cd] = Tree.SearchForClosestPoint(pt,       ... pt (in mesh c)
    %                                         prev_cp,  ... prev closest pt
    %                                         prev_ct,  ... prev triangle
    %                                         prev_cd)  ... prev distance
    %  Performs recursive search through the tree and returns
    %   cp       = Closest point on mesh to pt (both in mesh coordinates)
    %   ct       = Index in mesh of the triangle with the closest point
    %   cd       = Distance to closest point
    %
    % [cp,ct,cd] = LinearSearchForClosestPoint(T,pt,prev_cp,prev_ct,prev_cd)
    %   Performs linear search through tree, but returns same values
    %
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor
    
    properties
        Mesh        % handle for Mesh
        Tx          % index list of triangles
        Top         % pointer at top node
        minN  = 10  % minimum number of elements in a bucket
    end
    
    methods
        function Tree = TriangleCovTree(M)
            Tree.Mesh = M;
            N = size(M.Triangles,1);
            Tree.Tx = 1:N;
            Tree.Top = TriangleCovTreeNode(Tree,1,N);
        end
        
        function [cp,ct,cd] = LinearSearchForClosestPoint(T,pt,prev_cp,prev_ct,prev_cd)
            if isa(pt,'vct3Array')
                N = pt.NumEl();
                cp = vct3Array(N);
                ct = zeros(1,N);
                cd = zeros(1,N);
                for i=1:N
                    if nargin<3 
                        [ cp(i),ct(i),cd(i) ] = LinearSearchForClosestPoint(T,pt(i));
                    else
                        [ cp(i),ct(i),cd(i) ] = LinearSearchForClosestPoint(T,pt(i), ...
                                                      prev_cp(i),prev_ct(i),prev_cd(i));
                    end
                end
            else
                if nargin < 3
                    prev_ct = T.Tx(1);
                    [TV,p,q,r] = T.Mesh.Triangle(prev_ct);
                    [prev_cp,prev_cd] = TriangleClosestPoint(pt,p,q,r);
                end
                [cp,ct,cd] = LinearSearchForClosestPoint(T.Top,pt,prev_cp,prev_ct,prev_cd);
            end
        end
          
        function [cp,ct,cd] = SearchForClosestPoint(T,pt,prev_cp,prev_ct,prev_cd)
            if isa(pt,'vct3Array')
                N = pt.NumEl();
                cp = vct3Array(N);
                ct = zeros(1,N);
                cd = zeros(1,N);
                for i=1:N
                    if nargin<3 
                        [ cp(i),ct(i),cd(i) ] = SearchForClosestPoint(T,pt(i));
                    else
                        [ cp(i),ct(i),cd(i) ] = SearchForClosestPoint(T,pt(i), ...
                                                      prev_cp(i),prev_ct(i),prev_cd(i));
                    end
                end
            else
                if nargin < 3
                    prev_ct = T.Tx(1);
                    [TV,p,q,r] = T.Mesh.Triangle(prev_ct);
                    [prev_cp,prev_cd] = TriangleClosestPoint(pt,p,q,r);
                end
                [cp,ct,cd] = SearchForClosestPoint(T.Top,pt,prev_cp,prev_ct,prev_cd);
            end
        end
        
    end
    
end

